using System;
using System.Web;
using System.Web.Mvc;
using HIPS.Web.BusinessLogic.AssistedRegistration;
using HIPS.Web.Components.Cache;
using HIPS.Web.Data.Hips;
using HIPS.Web.Data.Hips.PcehrView;
using HIPS.Web.Data.Hips.Reference;
using HIPS.Web.Data.WebsiteDb;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.AssistedRegistration;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.PcehrView;
using HIPS.Web.UI.App_Start;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Properties;
using Microsoft.Web.Infrastructure.DynamicModuleHelper;
using Ninject;
using Ninject.Web.Common;
using Ninject.Web.Mvc.FilterBindingSyntax;
using WebActivator;

[assembly: WebActivator.PreApplicationStartMethod(typeof(NinjectWebCommon), "Start")]
[assembly: ApplicationShutdownMethod(typeof(NinjectWebCommon), "Stop")]

namespace HIPS.Web.UI.App_Start
{
    /// <summary>
    /// Startup module for Ninject.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
    /// </history>
    public static class NinjectWebCommon
    {

        #region Fields
        private static readonly Bootstrapper bootstrapper = new Bootstrapper();
        #endregion

        #region Methods

        /// <summary>
        /// Starts the application
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
        /// </history>
        public static void Start() 
        {
            DynamicModuleUtility.RegisterModule(typeof(OnePerRequestHttpModule));
            DynamicModuleUtility.RegisterModule(typeof(NinjectHttpModule));
            bootstrapper.Initialize(CreateKernel);
        }
        
        /// <summary>
        /// Stops the application.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
        /// </history>
        public static void Stop()
        {
            bootstrapper.ShutDown();
        }
        
        /// <summary>
        /// Creates the kernel that will manage your application.
        /// </summary>
        /// <returns>The created kernel.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
        /// </history>
        private static IKernel CreateKernel()
        {
            var kernel = new StandardKernel();
            kernel.Bind<Func<IKernel>>().ToMethod(ctx => () => new Bootstrapper().Kernel);
            kernel.Bind<IHttpModule>().To<HttpApplicationInitializationHttpModule>();
            
            RegisterServices(kernel);
            return kernel;
        }

        /// <summary>
        /// Load your modules or register your services here!
        /// </summary>
        /// <param name="kernel">The kernel.</param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
        /// </history>
        private static void RegisterServices(IKernel kernel)
        {
            //--- SettingsRepository ---//

            /* When asked to provide an ISettingsRepository use a CachedSettingsRepository */
            kernel.Bind<ISettingsRepository>()
                .To<CachedSettingsRepository>();

            /* When creating a CachedSettingsRepository use the MemoryCache for caching  */
            kernel.Bind<ICacheProvider>()
                .To<MemoryCacheProvider>()
                .WhenInjectedInto<CachedSettingsRepository>()
                .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.SettingsRepository_AbsoluteExpirationOffset);

            /* When creating a CachedSettingsRepository use a WebsiteDbRepository as the source ISettingsRepository */
            kernel.Bind<ISettingsRepository>()
                .To<WebsiteDbRepository>()
                .WhenInjectedInto<CachedSettingsRepository>();


            //--- HospitalRepository ---//

            /* When asked to provide an IHospitalRepository use a CachedHospitalRepository */
            kernel.Bind<IHospitalRepository>()
                .To<CachedHospitalRepository>();

            /* When creating a CachedHospitalRepository use the MemoryCache for caching */
            kernel.Bind<ICacheProvider>()
                .To<MemoryCacheProvider>()
                .WhenInjectedInto<CachedHospitalRepository>()
                .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.HospitalRepository_AbsoluteExpirationOffset);

            /* When creating a CachedHospitalRepository use a HospitalRepository as the source IHospitalRepository */
            kernel.Bind<IHospitalRepository>()
               .To<HospitalRepository>()
               .WhenInjectedInto<CachedHospitalRepository>();


            //--- AssistedRegistrationReferenceRepository ---//

            /* When asked to provide an IAssistedRegistrationReferenceRepository use a CachedAssistedRegistrationReferenceRepository */
            kernel.Bind<IAssistedRegistrationReferenceRepository>()
                .To<CachedAssistedRegistrationReferenceRepository>();

            /* When creating a CachedAssistedRegistrationReferenceRepository use the MemoryCache for caching */
            kernel.Bind<ICacheProvider>()
                .To<MemoryCacheProvider>()
                .WhenInjectedInto<CachedAssistedRegistrationReferenceRepository>()
                .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.AssistedRegistrationReferenceRepository_AbsoluteExpirationOffset);

            /* When creating a CachedAssistedRegistrationReferenceRepository use a WebsiteDbRepository as the source IAssistedRegistrationReferenceRepository */
            kernel.Bind<IAssistedRegistrationReferenceRepository>()
                .To<WebsiteDbRepository>()
                .WhenInjectedInto<CachedAssistedRegistrationReferenceRepository>();


            //--- PatientsWithoutPcehrRepository ---//

            /* When asked to provide an IPatientsWithoutPcehrRepository use a CachedPatientsWithoutPcehrRepository */
            kernel.Bind<IPatientsWithoutPcehrRepository>()
               .To<CachedPatientsWithoutPcehrRepository>();

            /* When creating a CachedPatientsWithoutPcehrRepository use the MemoryCache for caching */
            kernel.Bind<ICacheProvider>()
                .To<MemoryCacheProvider>()
                .WhenInjectedInto<CachedPatientsWithoutPcehrRepository>()
                .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.PatientsWithoutPcehrRepository_AbsoluteExpirationOffset); // TODO better key
               
            /* When creating a CachedPatientsWithoutPcehrRepository use a PatientWebServiceRepository as the source IPatientsWithoutPcehrRepository */
            kernel.Bind<IPatientsWithoutPcehrRepository>()
                .To<PatientWebServiceRepository>()
                .WhenInjectedInto<CachedPatientsWithoutPcehrRepository>();


            //--- AssistedRegistrationService ---//

            /* When asked to provide an IAssistedRegistrationService use a ValidatedAssistedRegistrationService */
            kernel.Bind<IAssistedRegistrationService>()
               .To<ValidatedAssistedRegistrationService>();

            /* When creating a ValidatedAssistedRegistrationService use an AssistedRegistrationService as the source IAssistedRegistrationService */
            kernel.Bind<IAssistedRegistrationService>()
                .To<AssistedRegistrationService>()
                .WhenInjectedInto<ValidatedAssistedRegistrationService>();


            //--- PatientRepository ---//

            //Need to retrieve the days discharged from settings so we can pass in to constructor.
            int daysDischarged = 0;
            using (WebsiteDbRepository settings = new WebsiteDbRepository())
            {
                daysDischarged = int.Parse(settings.GetSettings().GetSettingValue(Setting.SettingCodes.PcehrViewPatientDaysDischarged));
            }

            /* When asked to provide an IPatientRepository use a PatientRepository */
            kernel.Bind<IPatientRepository>()
                .To<PatientRepository>()
                .WithConstructorArgument("daysDischarged", daysDischarged);

            /* When creating a PatientRepository use the MemoryCache for caching */
            kernel.Bind<ICacheProvider>()
                .To<MemoryCacheProvider>()
                .WhenInjectedInto<PatientRepository>()
                .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.PatientRepository_AbsoluteExpirationOffset);


            //--- PcehrViewRepository ---//

            /* When asked to provide an IPcehrViewRepository use a PcehrViewRepository */
            kernel.Bind<IPcehrViewRepository>()
                .To<PcehrViewRepository>();

            /* When creating a PcehrViewRepository use the MemoryCache for caching */
            kernel.Bind<ICacheProvider>()
                .To<MemoryCacheProvider>()
                .WhenInjectedInto<PcehrViewRepository>()
                .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.PcehrViewRepository_AbsoluteExpirationOffset);


            //--- PcehrAccessFilter ---//

            /* When an action method has the PcehrAccessAttribute apply the PcehrAccessFilter */
            kernel.BindFilter<PcehrAccessFilter>(FilterScope.Action, 0)
                .WhenActionMethodHas<PcehrAccessAttribute>()
                .WithConstructorArgumentFromActionAttribute<PcehrAccessAttribute>("autoAccess", a => a.AutoAccess);
        }

        #endregion

    }
}
